<?php
/**
 * Top Ranking
 * https://webenginecms.org/
 * 
 * @version 1.1.0
 * @author Lautaro Angelico <http://lautaroangelico.com/>
 * @copyright (c) 2013-2019 Lautaro Angelico, All Rights Reserved
 * @build w3c8c718b75a0f1fa1a557f7f9d70877
 */

namespace Plugin\TopRanking;

class TopRanking {
	
	private $_configXml = 'config.xml';
	private $_modulesPath = 'modules';
	private $_cacheFile = 'rankings_top.cache';
	private $_cronFile = 'top_ranking.php';
	
	private $_resultsLimit = 50;
	private $_resetsBased = false;
	private $_classAvatarTemplateDir = 'character-avatars';
	private $_classAvatarStyle = '-moz-border-radius: 50%;-webkit-border-radius: 50%;border-radius: 50%;-khtml-border-radius: 50%;';
	private $_duelsWinStyle = 'color:#00cc00;';
	private $_duelsLoseStyle = 'color:#ff0000;';
	private $_classAvatarSize = 30;
	private $_gensImageSize = 40;
	private $_showResets = false;
	private $_showClass = true;
	private $_showKills = true;
	private $_showDuels = true;
	private $_showGens = true;
	private $_showGuild = true;
	private $_showOnlineStatus = true;
	private $_liveRankingData = false;
	private $_excludedPlayers;
	
	// CONSTRUCTOR
	
	function __construct() {
		global $custom;
		
		// load databases
		$this->mu = \Connection::Database('MuOnline');
		$this->db = \Connection::Database('Me_MuOnline');
		
		// vars
		$this->custom = $custom;
		
		// config file path
		$this->configFilePath = __PATH_TOPRANKING_ROOT__.$this->_configXml;
		if(!file_exists($this->configFilePath)) throw new \Exception(lang('topranking_error_5', true));
		$xml = simplexml_load_file($this->configFilePath);
		if(!$xml) throw new \Exception(lang('topranking_error_5', true));
		$this->_configs = convertXML($xml->children());
		if(!is_array($this->_configs)) throw new \Exception(lang('topranking_error_5', true));
		
		// set configs
		$this->_resultsLimit = $this->_configs['results_limit'];
		$this->_resetsBased = $this->_configs['resets_based'];
		$this->_classAvatarTemplateDir = $this->_configs['class_avatar_dir'];
		$this->_classAvatarSize = $this->_configs['class_avatar_size'];
		$this->_gensImageSize = $this->_configs['gens_image_size'];
		$this->_showResets = $this->_configs['show_resets'];
		$this->_showClass = $this->_configs['show_class'];
		$this->_showKills = $this->_configs['show_kills'];
		$this->_showDuels = $this->_configs['show_duels'];
		$this->_showGens = $this->_configs['show_gens'];
		$this->_showGuild = $this->_configs['show_guild'];
		$this->_showOnlineStatus = $this->_configs['show_online'];
		$this->_liveRankingData = $this->_configs['live_ranking'];
		if(check_value($this->_configs['excluded_players'])) {
			$excludedPlayers = explode(",", $this->_configs['excluded_players']);
			$this->_excludedPlayers = $excludedPlayers;
		}
		
		// cron
		$this->_checkCron();
	}
	
	// PUBLIC FUNCTIONS
	
	public function loadModule($module) {
		if(!\Validator::Alpha($module)) throw new \Exception(lang('topranking_error_4', true));
		if(!$this->_moduleExists($module)) throw new \Exception(lang('topranking_error_4', true));
		if(!@include_once(__PATH_TOPRANKING_ROOT__ . $this->_modulesPath . '/' . $module . '.php')) throw new \Exception(lang('topranking_error_4', true));
	}
	
	public function updateCache() {
		
		$rankingData = $this->_getRankingData();
		if(!is_array($rankingData)) {
			$this->_updateCacheFile("");
			return;
		}
		
		$rankingDataJson = $this->_prepareCacheData($rankingData);
		if(!check_value($rankingDataJson)) {
			$this->_updateCacheFile("");
			return;
		}
		
		$this->_updateCacheFile($rankingDataJson);
	}
	
	public function getRankingData() {
		if($this->_liveRankingData) {
			return $this->_getRankingData();
		} else {
			$cacheData = $this->_loadCacheData();
			if(!is_array($cacheData)) return;
			return $cacheData;
		}
	}
	
	public function getClassImage($class) {
		if(!array_key_exists($class, $this->custom['character_class'])) return 'x';
		if(!check_value($this->custom['character_class'][$class][2])) return 'x';
		
		$imagePath = __PATH_TEMPLATE_IMG__ . $this->_classAvatarTemplateDir . '/' . $this->custom['character_class'][$class][2];
		$className = $this->custom['character_class'][$class][0];
		
		return '<img src="'.$imagePath.'" width="auto" height="'.$this->_classAvatarSize.'px" data-toggle="tooltip" data-placement="top" title="'.$className.'" alt="'.$className.'" style="'.$this->_classAvatarStyle.'"/>';
	}
	
	public function getGensImage($type=null) {
		if(is_null($type)) return '<span style="display:inline-block;width:auto;height:'.$this->_gensImageSize.'px;"></span>';
		if($type == 1) {
			return '<img src="'.__PATH_TEMPLATE_IMG__.'gens_1.png" width="auto" height="'.$this->_gensImageSize.'px" data-toggle="tooltip" data-placement="top" title="'.lang('rankings_txt_26',true).'" alt="'.lang('rankings_txt_26',true).'"/>';
		} else {
			return '<img src="'.__PATH_TEMPLATE_IMG__.'gens_2.png" width="auto" height="'.$this->_gensImageSize.'px" data-toggle="tooltip" data-placement="top" title="'.lang('rankings_txt_27',true).'" alt="'.lang('rankings_txt_27',true).'"/>';
		}
	}
	
	public function showResets() {
		return $this->_showResets;
	}
	
	public function showClass() {
		return $this->_showClass;
	}
	
	public function showKills() {
		return $this->_showKills;
	}
	
	public function showDuels() {
		return $this->_showDuels;
	}
	
	public function showGens() {
		return $this->_showGens;
	}
	
	public function showGuild() {
		return $this->_showGuild;
	}
	
	public function showOnlineStatus() {
		return $this->_showOnlineStatus;
	}
	
	public function getDuelsWinStyle() {
		return $this->_duelsWinStyle;
	}
	
	public function getDuelsLoseStyle() {
		return $this->_duelsLoseStyle;
	}
	
	public function isOnline($player) {
		$Character = new \Character();
		$characterData = $Character->CharacterData($player);
		if(!is_array($characterData)) return;
		
		$result = $this->db->query_fetch_single("SELECT * FROM "._TBL_MS_." WHERE "._CLMN_CONNSTAT_." = ? AND "._CLMN_MS_MEMBID_." = ?", array(1, $characterData[_CLMN_CHR_ACCID_]));
		if(!is_array($result)) return;
		
		$characterIDC = $Character->AccountCharacterIDC($characterData[_CLMN_CHR_ACCID_]);
		if(!check_value($characterIDC)) return;
		
		if($characterIDC != $characterData[_CLMN_CHR_NAME_]) return;
		return true;
	}
	
	// PRIVATE FUNCTIONS
	
	private function _moduleExists($module) {
		if(!check_value($module)) return;
		if(!file_exists(__PATH_TOPRANKING_ROOT__ . $this->_modulesPath . '/' . $module . '.php')) return;
		return true;
	}
	
	private function _prepareCacheData($data) {
		if(!is_array($data)) return;
		return json_encode($data);
	}
	
	private function _updateCacheFile($data) {
		$file = __PATH_CACHE__.$this->_cacheFile;
		if(!file_exists($file)) return;
		if(!is_writable($file)) return;
		
		$fp = fopen($file, 'w');
		fwrite($fp, $data);
		fclose($fp);
		return true;
	}
	
	private function _loadCacheData() {
		$file = __PATH_CACHE__.$this->_cacheFile;
		if(!file_exists($file)) return;
		
		$cacheData = file_get_contents($file);
		if(!check_value($cacheData)) return;
		
		$cacheDataArray = json_decode($cacheData, true);
		if(!is_array($cacheDataArray)) return;
		
		return $cacheDataArray;
	}
	
	private function _checkCron() {
		$result = $this->db->query_fetch_single("SELECT * FROM ".WEBENGINE_CRON." WHERE cron_file_run = ?", array($this->_cronFile));
		if(is_array($result)) return;
		$this->_createCron();
	}
	
	private function _createCron() {
		if(!file_exists(__PATH_CRON__ . $this->_cronFile)) throw new \Exception(lang('topranking_error_3', true));
		$cronMd5 = md5_file(__PATH_CRON__ . $this->_cronFile);
		if(!check_value($cronMd5)) throw new \Exception(lang('topranking_error_3', true));
		$insertData = array(
			'Top Ranking',
			$this->_cronFile,
			300,
			1,
			0,
			$cronMd5
		);
		$result = $this->db->query("INSERT INTO ".WEBENGINE_CRON." (cron_name, cron_file_run, cron_run_time, cron_status, cron_protected, cron_file_md5) VALUES (?, ?, ?, ?, ?, ?)", $insertData);
		if(!$result) throw new \Exception(lang('topranking_error_3', true));
	}
	
	private function _getRankingData() {
		
		if($this->_resetsBased) {
			// ranking based on resets
			$result = $this->mu->query_fetch("SELECT TOP ".$this->_resultsLimit." *, ("._CLMN_CHR_LVL_." + "._CLMN_ML_LVL_.") AS combinedLevel FROM "._TBL_CHR_." WHERE "._CLMN_CHR_NAME_." NOT IN(".$this->_excludedPlayersArray().") ORDER BY "._CLMN_CHR_RSTS_." DESC, combinedLevel DESC, "._CLMN_ML_EXP_." DESC, "._CLMN_CHR_PK_KILLS_." DESC, id ASC");
		} else {
			// ranking based on level + master level
			$result = $this->mu->query_fetch("SELECT TOP ".$this->_resultsLimit." *, ("._CLMN_CHR_LVL_." + "._CLMN_ML_LVL_.") AS combinedLevel FROM "._TBL_CHR_." WHERE "._CLMN_CHR_NAME_." NOT IN(".$this->_excludedPlayersArray().") ORDER BY combinedLevel DESC, "._CLMN_ML_EXP_." DESC, "._CLMN_CHR_PK_KILLS_." DESC, id ASC");
		}
		
		
		foreach($result as $row) {
			
			// Gens
			$gensType = null;
			if($this->showGens()) {
				$gensData = $this->mu->query_fetch_single("SELECT * FROM "._TBL_GENS_." WHERE "._CLMN_GENS_NAME_." = ?", array($row[_CLMN_CHR_NAME_]));
				if(is_array($gensData)) {
					$gensType = $gensData[_CLMN_GENS_TYPE_];
				}
			}
			
			// Guild
			$guildName = null;
			if($this->showGuild()) {
				$guildData = $this->mu->query_fetch_single("SELECT * FROM "._TBL_GUILDMEMB_." WHERE "._CLMN_GUILDMEMB_CHAR_." =?", array($row[_CLMN_CHR_NAME_]));
				if(is_array($guildData)) {
					$guildName = $guildData[_CLMN_GUILDMEMB_NAME_];
				}
			}
			
			$players[] = array(
				'name' => $row[_CLMN_CHR_NAME_],
				'level' => $row[_CLMN_CHR_LVL_],
				'class' => $row[_CLMN_CHR_CLASS_],
				'map' => $row[_CLMN_CHR_MAP_],
				'kills' => $row[_CLMN_CHR_PK_KILLS_],
				'resets' => $row[_CLMN_CHR_RSTS_],
				'masterlevel' => $row[_CLMN_ML_LVL_],
				'duelswon' => $row['WinDuels'],
				'duelslost' => $row['LoseDuels'],
				'combinedLevel' => $row['combinedLevel'],
				'gens' => $gensType,
				'guild' => $guildName,
			);
		}
		
		return $players;
	}
	
	private function _excludedPlayersArray() {
		if(!is_array($this->_excludedPlayers)) return "''";
		$return = array();
		foreach($this->_excludedPlayers as $characterName) {
			$return[] = "'".$characterName."'";
		}
		return implode(",", $return);
	}
	
}